/*
 * Copyright (C) 2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include <span>
#include <wtf/AbstractThreadSafeRefCountedAndCanMakeWeakPtr.h>
#include <wtf/NativePromise.h>
#include <wtf/ThreadSafeWeakPtr.h>

namespace WebCore {

class Exception;
class ReadableStreamSource;
class ScriptExecutionContext;
class WebTransportBidirectionalStream;
class WebTransportSendStream;
class WebTransportSessionClient;

struct WebTransportConnectionStats;
struct WebTransportReceiveStreamStats;
struct WebTransportSendGroupIdentifierType;
struct WebTransportSendStreamStats;
struct WebTransportStreamIdentifierType;

using WebTransportSendGroupIdentifier = ObjectIdentifier<WebTransportSendGroupIdentifierType>;
using WebTransportStreamIdentifier = ObjectIdentifier<WebTransportStreamIdentifierType>;
using WebTransportStreamPromise = NativePromise<WebTransportStreamIdentifier, void>;
using WebTransportSendPromise = NativePromise<std::optional<Exception>, void>;
using WebTransportConnectionStatsPromise = NativePromise<WebTransportConnectionStats, void>;
using WebTransportSendStreamStatsPromise = NativePromise<WebTransportSendStreamStats, void>;
using WebTransportReceiveStreamStatsPromise = NativePromise<WebTransportReceiveStreamStats, void>;

using WebTransportSessionErrorCode = uint32_t;
using WebTransportStreamErrorCode = uint64_t;

class WEBCORE_EXPORT WebTransportSession : public AbstractThreadSafeRefCountedAndCanMakeWeakPtr {
public:
    virtual ~WebTransportSession();

    virtual Ref<WebTransportSendPromise> sendDatagram(std::optional<WebTransportSendGroupIdentifier>, std::span<const uint8_t>) = 0;
    virtual Ref<WebTransportStreamPromise> createOutgoingUnidirectionalStream() = 0;
    virtual Ref<WebTransportStreamPromise> createBidirectionalStream() = 0;
    virtual Ref<WebTransportSendPromise> streamSendBytes(WebTransportStreamIdentifier, std::span<const uint8_t>, bool withFin) = 0;
    virtual Ref<WebTransportConnectionStatsPromise> getStats() = 0;
    virtual Ref<WebTransportSendStreamStatsPromise> getSendStreamStats(WebTransportStreamIdentifier) = 0;
    virtual Ref<WebTransportReceiveStreamStatsPromise> getReceiveStreamStats(WebTransportStreamIdentifier) = 0;
    virtual Ref<WebTransportSendStreamStatsPromise> getSendGroupStats(WebTransportSendGroupIdentifier) = 0;

    virtual void cancelReceiveStream(WebTransportStreamIdentifier, std::optional<WebTransportStreamErrorCode>) = 0;
    virtual void cancelSendStream(WebTransportStreamIdentifier, std::optional<WebTransportStreamErrorCode>) = 0;
    virtual void destroyStream(WebTransportStreamIdentifier, std::optional<WebTransportStreamErrorCode>) = 0;
    virtual void terminate(WebTransportSessionErrorCode, CString&&) = 0;
    virtual void datagramIncomingMaxAgeUpdated(std::optional<double>) = 0;
    virtual void datagramOutgoingMaxAgeUpdated(std::optional<double>) = 0;
    virtual void datagramIncomingHighWaterMarkUpdated(double) = 0;
    virtual void datagramOutgoingHighWaterMarkUpdated(double) = 0;
};

}
